/*
 *     This file is part of the Jira Agile synchronization connector for Squash TM (plugin.requirement.xsquash4jira) project.
 *     Copyright (C) 2017 - 2019 Henix, henix.fr - All Rights Reserved
 *
 *     Unauthorized copying of this file, via any medium is strictly prohibited
 *     Proprietary and confidential
 *
 * 	 (C)Henix. Tous droits réservés.
 *
 * 	Avertissement : ce programme est protégé par la loi relative au droit d'auteur et par les conventions internationales. Toute reproduction ou distribution partielle ou totale du logiciel, par quelque moyen que ce soit, est strictement interdite.
 */
/*
 * 1/ The Main view
 * ----------------
 * 
 * This is the main view. It supervises the following elements :
 * 
 * - the SideBar, on the left
 * - the ButtonPane, at the bottom
 * - indirectly, the screens on the center. It actually delegates to the ScreenManager
 * - the main model, aka MegaModel, which hold every important business data.
 * 
 * It also centralizes the event handling coming from those various parts and dispatches them, instead of 
 * letting those actors listening to each others and react by themselves. It was done so because it makes 
 * the code more explicit and ensures that the right code is run in the right order. See the 'events' hash 
 * for details.
 * 
 * The ScreenManager, ButtonPane and SideBar are fairly simple and require no more explanations. 
 * 
 * 
 * 2/ Stages and Screen Flow
 * -------------------------
 * 
 * Let call each step of the process a Stage. A stage is ready when the required prerequisite data are available 
 * to work with, and is completed when the user entered its selection/choices/preferences for the current stage.
 * Very often the completion of a stage means that the next stage is available because it is likely to be the
 * prerequisite data for that next stage.
 * 
 * 
 * The ScreenFlow defines the transitions from one screen to the next, back and forth, and which transitions are legal
 * according to the state of the MegaModel. It also manages the status of the various screen (to some extents) and also 
 * knows which buttons should be displayed depending on whether the current screen is the first of the flow, the last, 
 * or any other.
 * 
 * The ordered, linear set of transition that define which screen can be visited and in which order is called a route. 
 * Some properties of the MegaModel have a significant impact on the route. When one of those properties change the MainView 
 * instructs the ScreenFlow to compute a new route which is then dispatched to the ScreenManager and the SideBar. 
 *  
 * It is used by many other actors, notably the ScreenManager. 
 * 
 * 
 * 3/ Models and Screens
 * ---------------------
 * 
 * The screens show the building stages of the test plan, starting from the WelcomeScreen and ending at the  
 * DoneScreen (although technically the user never interact with it directly).
 * Each Screen has its own model. Each Screen extends BaseScreen, and each Model extends BaseModel, so that 
 * they also inherit their logic regarding status, model readiness etc.
 * 
 * 
 * All UI-related events are handled by the screen directly. Those events won't leak to the MainView and are 
 * thus are not in scope of the global events treated by the main view. On the other hand, status-related 
 * events will be forwarded to the MainView in order to trigger an update for other parts of the UI. 
 *
 *
 * A screen is responsible to know its status. The status depends on the data present in its model, and to 
 * some extents the status of the screens that comes before in the current route. The status list and their 
 * logic are explained in screens/BaseScreen.js.
 *
 *
 * Each of these screens has its own model (excepted in two occurrences, see below). Although local to their 
 * screens, the model also keep alive a reference to the MegaModel. The intent of that structure is to 
 * keep concerns separated : 
 * 	a - the state of the view is managed by the local model (which checkboxes are ticked for instance), 
 * so that they won't pollute the MegaModel,
 * 	b - the business data that are important to the overall process are forwarded to the MegaModel.
 * 
 * 
 * The only exception to the rule 'one screen - one model' is for the subroute 'ByRelease' and 'BySprint', 
 * which both consists of two screens sharing the same model. They exchange so many internal state 
 * that having them communicating via the MegaModel would have been more confusing than useful. 
 * 
 * 
 * The Models and the MegaModel are tightly integrated : the models exposes direct getter/setters to the 
 * MegaModel properties, and listen to changes in their prerequisite data held by the  MegaModel 
 * (see Stages and ScreenFlow). More on this in screens/BaseModel.js
 * 
 * 
 * @see screens/BaseScreen
 * @see screens/BaseModel
 * @see screens/MegaModel
 * 
 * 
 * 4/ MegaModel
 * ------------
 * 
 * 	The MegaModel holds the business properties that are essential to the creation of the execution plan,
 * while the UI-state details are handled by private screen models (see Models and Screens). It is 
 * created once by the MainView and then passed to every other parties so that it serves as a centralized 
 * source of truth. The file screens/MegaModel describe its attributes. 
 * 
 *  Almost every properties of the MegaModel are under scrutiny because they drive the route, 
 *  the screen status, and the stage readiness/completion (see Stages and ScreenFlow) : 
 *  
 *  - 'action' and 'strategy' have a direct influence on the route, and are directly monitored by the main view.
 *  - (almost) every property is the output of stage 'n' and the prerequisite for stage 'n+1'. The screen models
 *  listen to their change because a/ it can make a stage available and b/ invalidate a stage that was completed 
 *  until now because the prerequisite data have changed.
 *  
 *  @see screens/MegaModel
 *   
 *  
 * 
 */
define(["jquery", "handlebars", "backbone", "underscore", "squash.basicwidgets",
	"./JirasyncExecplanScreenFlow", "./screens/MegaModel", "./screens/JirasyncExecplanScreens", "./screens/ModelEventsMixin", 
	"jquery.squash.formdialog"], 
	function($, Handlebars, Backbone, _, widgets, ScreenFlow, MegaModel, screens, ModelEventsMixin){


	
    /* ************************************************************
     * 
     * The screen manager  
     * 
     ************************************************************* */
	
	
	var ScreenManager = Backbone.View.extend({
		
		el : "#mainview",
		
		screens : [],
		
		screenMap : {},
		
		initialize : function(options){
			this.mega = options.mega;
			this.screens = options.screens;		
			
			var screenMap = {};
			options.screens.forEach(function(screen){
				screenMap[screen.name] = screen;
			});
			this.screenMap = screenMap;
		},
		
		renderScreen : function(screen){
			this.screenMap[screen].renderIfRequired();
		},

		renderAll : function(){
			this.screens.forEach(function(screen){
				screen.renderIfRequired();
			});
		},
		
		/*
		 * Screens slide (fold) to the left when the screen flow advance, 
		 * and to the right when moving back. This is done with the class 
		 * 'slide-left'.
		 * 
		 *  So basically every screens left to the new screen name
		 *  must have class 'slide-left', and from 'screen name' and 
		 *  forward the class needs to be removed.
		 */
		showScreen : function(screenName){
			
    		// render all screens (if required) because 
    		// one might scroll by a few screen 
    		this.renderAll();
			
			var idx = _.findIndex(this.screens, function(screen){
				return screen.name === screenName;
			});
			
			this.screens.slice(0,idx).forEach(function(scr){
				scr.addClass('slide-left');
			});
			
			this.screens.slice(idx).forEach(function(scr){
				scr.removeClass('slide-left');
			});
						
		},
		
		updateRoute : function(newRoute){
			this.route = newRoute;
			
			this.screens.forEach(function(screen){
				if (_.contains(newRoute, screen.name)){
					screen.show();
				}
				else{
					screen.hide();
				}
			});
		}
	});

    
    /* ***************************************
     * 
     * The button pane
     * 
     * **************************************/
    
    var ButtonPane = Backbone.View.extend({
    	
    	el : "#buttonpane",
    	
    	btnSpecs : [],
    	
    	initialize : function(){
    		var self = this;
    	},
    	
    	setBtnSpecs : function(newSpecs){
    		this.btnSpecs = newSpecs;
    	},
    	
    	render : function(){
    		var container = this.$el;
    		container.empty();
    		
    		this.btnSpecs.forEach(function(btn){
    			var label = btn.label;
    			var event = btn.event;
    			container.prepend('<input class="sq-btn" value="'+label+'" data-event="'+event+'"></input>');
    		});
    		
    		return this;
    	}
    	
    });
    
    /* **************************************************
     * 
     * The sidebar
     * 
     * **************************************************/
    
    var SideBar = Backbone.View.extend({
    	
    	el : '#sidebar',
    	
    	steps : [],
    	
    	visited : {},
    	
    	initialize : function(){
    		var self = this;
    		var $steps = this.$el.find('.step');
    		this.steps = $steps.get();

    		$steps.each(function(){
    			var name = $(this).data('screen');
    			self.visited[name] = false;
    		});
    	},
    	
    	
    	/*
    	 * Setting the active screen means also to update the 
    	 * css of the other items :
    	 * 
    	 * 1/ sidebar items that are located before the active 
    	 * screen show their real status
    	 * 
    	 * 2/ items located after show their real status 
    	 * only if they have been visited before, otherwise 
    	 * they show blank (the css class 'not-reached').
    	 * 
    	 * We must then keep track of which has been visited and 
    	 * which are not.
    	 * 
    	 */
    	setActiveScreen : function(screen){
    		
    		this.$el.find('.step').removeClass('active');
    		
    		var $step = this.$el.find('#sidebar-'+screen);
    		$step.addClass('active');
    		
    		var self = this;

    		// remove not-reached for the active step and all steps before, 
    		// and also take note of which is considered visited
    		$step.prevAll().addBack().each(function(){
    			var $s = $(this);
    			var name = $s.data('screen');
    			
    			$s.removeClass('not-reached');
    			if (! $s.hasClass('premature') ){
    				self.visited[name] = true;
    			}
    		});
    		
    		// add 'not-reached' for all steps after, unless 
    		// they were visited once
    		$step.nextAll().each(function(){
    			var $s = $(this);
    			var name = $s.data('screen');
    			
    			if (self.visited[name] === false){
    				$s.addClass('not-reached');
    			}
    		});
    		
    	},
    	
    	updateRoute : function(route){
    		
    		this.steps.forEach(function(step){
    			var $step = $(step);
    			var name = $(step).data('screen');
    			if (_.contains(route, name)){
    				$step.show();
    			}
    			else{
    				$step.hide();
    			}
    		});
    		
    	},
    	
    	updateStatus : function(evt){
    		var step = this.$el.find('#sidebar-'+evt.screen);
    		var clazz = evt.status.toLowerCase().replace('_','-');
    		step.removeClass('premature ready complete needs-review');
    		step.addClass(clazz);
    		
    	}
    	
    	
    	
    });
    
   
    /* **************************************************
     * 
     * The main view
     * 
     * actually it does pretty much nothing on its own, 
     * it just initializes everything.
     * 
     * **************************************************/
    

    var Master = Backbone.View.extend({
    	
    	el : "#jirasync-wizard-pane",
    	
    	screenFlow : null,
    	screenManager : null,
    	buttonPane : null,
    	    
    	initialize : function(options){
    		
    		var mega = new MegaModel(options);
    		this.model = mega;
    		
    		this.initScreens();
    		
    		var conf = {
    			mega : mega,
    			screens : this.screens
			};
    		
    		this.screenFlow = new ScreenFlow({}, conf);
    		this.sideBar = new SideBar();
    		this.screenManager = new ScreenManager(conf);
    		this.buttonPane = new ButtonPane(conf);
    	
    		// this one comes from the ModelEventsMixin
    		this.bindModelEvents();
    		
    	},
    	
    	render : function(){
    		this.changeRoute();
    		var first = this.screenFlow.firstScreen;
    		this.activateScreen(first);
    	},
    	
    	activateScreen : function(screen){
    		
    		this.screenFlow.setActiveScreen(screen);
    		this.screenFlow.refreshStatus();
    		
    		var btns = this.screenFlow.getActiveButtons();
    		
    		this.buttonPane.setBtnSpecs(btns);
    		this.buttonPane.render();
    		
    		this.screenManager.renderAll();
    		this.screenManager.showScreen(screen);
    		    		    		
    	},
    	
    	
    	// ***************** events **********************
    	
    	events : {
    		'click #buttonpane input' 	: 'navigate',
    		'click #sidebar .step'		: 'teleport',
    		'model-invalidated'			: 'refreshStatus',
    		'screenstatus-changed'		: 'statusChanged',
    	},
    	
    	modelEvents : {
    		'change:action' : 'changeRoute',
    		'change:strategy' : 'changeRoute'
    	},
    	
    	// **************** navigation ********************
    	
    	
    	activateScreen : function(screen){
    		this.screenFlow.setActiveScreen(screen);
    		this.screenFlow.refreshStatus();
    		this.sideBar.setActiveScreen(screen);    		
    		
    		var btns = this.screenFlow.getActiveButtons();    		
    		
    		this.buttonPane.setBtnSpecs(btns);
    		this.buttonPane.render();
    		
    		this.screenManager.showScreen(screen);
    		    		    		
    	},
    	
    	
    	navigate : function(evt){
    		var event = $(evt.currentTarget).data('event');
    		var newScreen;
    		switch(event){
    		case "advance" :
    		case "go" :
    			newScreen = this.screenFlow.advance();
    			break;
    		case "back" : 
    			newScreen = this.screenFlow.back();
    			break;    			
    		}
    			
    		this.activateScreen(newScreen);
    	},
    	
    	teleport : function(evt){
    		var screen = $(evt.currentTarget).data('screen');
    		this.activateScreen(screen);
    	},
    	
    	
    	// *************** route and status handler **************
    	
    	changeRoute : function(){
    		var route = this.screenFlow.defineRoute();
    		this.sideBar.updateRoute(route);
    		this.screenManager.updateRoute(route);
    		this.screenFlow.refreshStatus();
    	},
    	
    	statusChanged : function(evt, data){
    		this.sideBar.updateStatus(data);
    	},
    	
    	refreshStatus : function(evt, data){
    		this.screenFlow.refreshStatus(data.screen);
    	},
    	
    	// ************* init code **************************
		
    	debugScreenStatus : function(){
    		this.screens.forEach(function(scr){
    			console.log(scr.name + " : status "+scr.status+", visited : "+scr.visited);
    		});
    	},
    	
		initScreens : function(){
				    	
			var modelconf = { mega : this.model };
			
			// create all the models
			// since they receive no attributes, the first parameter is empty.
			
			var actionModel = new screens.ActionModel({}, modelconf);
			var strategyModel = new screens.StrategyModel({}, modelconf);
			var byreleaseModel = new screens.ByReleaseModel({}, modelconf);
			var bysprintModel = new screens.BySprintModel({}, modelconf);
			var jqlModel = new screens.JQLModel({}, modelconf);
			var ticketselectModel = new screens.TicketSelectionModel({}, modelconf);
			var testcaseselectModel = new screens.TestCaseSelectionModel({}, modelconf);
			var iterationModel = new screens.IterationModel({}, modelconf);
			var doneModel = new screens.DoneModel({}, modelconf);
			
			
			// create the view. Remember that some views share the same model
			this.screens = [
				new screens.ActionScreen({model : actionModel}),
				new screens.StrategyScreen({model : strategyModel}),
				new screens.ProjectSelectionScreen({model : byreleaseModel}),
				new screens.ReleaseSelectionScreen({model : byreleaseModel}),
				new screens.BoardSelectionScreen({model : bysprintModel}),
				new screens.SprintSelectionScreen({model : bysprintModel}),
				new screens.JQLScreen({model : jqlModel}),
				new screens.TicketSelectionScreen({model : ticketselectModel}),
				new screens.TestCaseSelectionScreen({model : testcaseselectModel}),
				new screens.IterationScreen({model : iterationModel}),
				new screens.DoneScreen({model : doneModel})
			];


			
			return this.screens;
		}

    });
    
    _.extend(Master.prototype, ModelEventsMixin);
    
    
    // ************* the premature back button ****************
    
    var backpop = $("#back-popup");
    backpop.formDialog();
    
    backpop.on('formdialogconfirm', function(){
    	window.history.back();
    });
    
    backpop.on('formdialogcancel', function(){
    	backpop.formDialog('close');
    });
    
    $("#back").on('click', function(){
    	backpop.formDialog('open');
    });
    
    

    return Master;

});