/*
 *     This file is part of the Jira Agile synchronization connector for Squash TM (plugin.requirement.xsquash4jira) project.
 *     Copyright (C) 2017 - 2019 Henix, henix.fr - All Rights Reserved
 *
 *     Unauthorized copying of this file, via any medium is strictly prohibited
 *     Proprietary and confidential
 *
 * 	 (C)Henix. Tous droits réservés.
 *
 * 	Avertissement : ce programme est protégé par la loi relative au droit d'auteur et par les conventions internationales. Toute reproduction ou distribution partielle ou totale du logiciel, par quelque moyen que ce soit, est strictement interdite.
 */
/* ************************************************************
 * 
 * ScreenFlow
 * 
 * See JirasyncExecplanMainView for an introduction to the 
 * tasks of the ScreenFlow.
 * 
 * This is the central UI State model dedicated to the screen
 * flow. Its not really a router (no history involved), nor 
 * a model (no business data maintained). We still make it a 
 * Model for the syntactic sugar that comes with it.
 * 
 * This object listen to two major properties of the MegaModel : 
 * - the action (what to do with the selected node),
 * - the strategy (from which source the JIRA tickets will 
 * 		be retrieved).
 * 
 * 
 * 
 * The ButtonPane will ask for which buttons it should display
 * each time a new screen becomes active. The ScreenFlow 
 * will reply with button specification such as this : 
 * 
 *  [{
 *  	label : 'value (the text to display)',
 *  	event : 'event that should be fired'
 *  }]
 *  
 *  example : 
 *  [
 *  	{ 
 *  		label : 'previous',
 *  		event : 'back' 
 *  	},
 *   	{
 *   		label : 'next',
 *   		event : 'advance'
 *   	}
 *  ]
 * 
 * ************************************************************/
define(["backbone", "underscore", "squash.translator"], function(Backbone, _, translator){
	

	
	/* ******************************************
	 * A note about the mappings : 
	 * 
	 * Each mapping states : 
	 * 	- which view comes before,
	 * 	- which view comes next.
	 *  
	 *  Acceptable value are either the view name (for simple case), or a mapping object
	 *  (when things are more complex). The said object may declare a mapping between the 
	 *  value of one of the model attribute and screen name. This allows for branching 
	 *  depending on user choice. An example is worth thousand lines : 
	 *  
	 *  {
	 *  	action : {
	 *  		"action=UPDATE_IT" : ViewOne,
	 *  		"action=UPDATE_CPG" : ViewTwo,
	 *  		"else" : ViewThree
	 *  	}
	 *  
	 *  }
	 *  
	 *  This means that if the model attribute 'action' has the value 'UPDATE_IT', 
	 *  then the next view will be ViewOne, 'UPDATE_CPG' leads to ViewTwo and 
	 *  in any other case this will be ViewThree.
	 *  
	 *  Taken together those mappings effectively forms a loose graph.
	 * ******************************************/
	
	class Mapping{
		constructor(previous, next){
			this.previous = previous;
			this.next = next;
		}
	}
	
	/*
	 * The ScreenFlow object itself
	 */
	
	var ScreenFlow = Backbone.Model.extend({
		
		// ********* attributes **************
		
		firstScreen : "strategy",
		
		activeScreen : "strategy", 
		
		screenMap : {},
		
		/*
		 * Define which screen is connected to which other screen, and under which conditions.
		 * These mappings may be altered by method "initFlow", see below.
		 */
		mappings : {
			
			"strategy" : 		new Mapping(null, {								
									"strategy=BY_RELEASE" : "project-select",
									"strategy=BY_SPRINT" : "board-select",
									"strategy=BY_JQL"	: "jql",
									"else" : "not implemented !",
								}),		
								
			// BY_RELEASE strategy screens
			"project-select" :	new Mapping("strategy", "release-select"),
			"release-select" :	new Mapping("project-select", "ticket-select"),
			
			
			// BY_SPRINT strategy screens
			"board-select" :	new Mapping("strategy", "sprint-select"),
			"sprint-select" :	new Mapping("board-select", "ticket-select"),
			
			// BY_JQL strategy screen
			"jql" :				new Mapping("strategy", "ticket-select"),
			
			// merge back the graph flow
			"ticket-select" : 	new Mapping({
									"strategy=BY_RELEASE" : "release-select",
									"strategy=BY_SPRINT" : "sprint-select",
									"strategy=BY_JQL"	 : "jql",
									"else" :		"not implemented !"								
								}, "testcase-select"),								

			"testcase-select" : new Mapping("ticket-select", {
									"action=CREATE_IT" : "iteration-create",
									"else" : "done"
								}),

			"iteration-create" : new Mapping("testcase-select", "done"),
								
			"done" : 			new Mapping({
									"action=CREATE_IT" : "iteration-create",
									"else" : "testcase-select"
								}, null)
						
		},
			
		// *************** event binding ***************************
		
		initialize : function(attr, options){
			
			this.mega = options.mega;

			this.initFlow(options);
			
			this.initScreenMap(options);
			
		},
		
		initFlow : function(options){
			var nodeType = this.mega.selectedNode().type;
			switch(nodeType){
			
			case 'iteration' : break;
				
			case 'campaign' :
				// if the node type is "campaign", insert "action" at the head of the flow
				this.mappings["action"] = new Mapping(null, "strategy");
				this.mappings["strategy"].previous = "action";
				this.firstScreen = "action";
				break;
			default : throw "unsupported node type : "+nodeType;
			}
		},
		
		initScreenMap : function(options){
			// map the screens by name
			var screenMap = {};
			options.screens.forEach(function(screen){
				screenMap[screen.name] = screen;
			});
			
			this.screenMap = screenMap;
		},

		
		// ********************* screen and route management *****************
		
		
		defineRoute : function(){
			// we unfold the mappings, starting from the firstScreen, and collect the 
			// view names that will be traversed.
			var scr = this.firstScreen;
			var screens = [scr];
			while (scr !== null){
				scr = this._next(scr);
				if (scr !== null){
					screens.push(scr);
				}
			}
			
			return screens;
		},
		
		setActiveScreen : function(screenName){
			this.activeScreen = screenName;
			// disable all screens except the active screen
			_.values(this.screenMap).forEach(function(screen){
				screen.setActive(false);
			})
			
			this.screenMap[screenName].setActive(true);
			
			return this.activeScreen;
		},
		
		/*
		 * See method refreshStatus of the screens 
		 * to know more about how status is defined.
		 */
		refreshStatus : function(scrName){

			
			if (scrName === undefined){
				// refresh for all

				var route = this.defineRoute();
				var predecessorStatus = undefined;
				var screenName;

				var prevStatus = "COMPLETE";
				do{
					var screenName = route.shift();
					var screen = this.screenMap[screenName];
					prevStatus = screen.refreshStatus(prevStatus);				
				}
				while(route.length > 0);
			}
			else{
				// refresh only the said screen
				
				var prev = this._prev(scrName);
				var prevScreen = this.screenMap[prev];
				var prevStatus = prevScreen.status;
				
				var screen = this.screenMap[scrName];
				screen.refreshStatus(prevStatus);
			}
			
		},
				
		advance : function(){
			this.activeScreen = this._next(this.activeScreen);
			return this.activeScreen;
		},
		
		back : function(){
			this.activeScreen = this._prev(this.activeScreen);
			return this.activeScreen;
		},
		
		_next : function(screen){
			var nextMapping = this.mappings[screen].next;
			return this.resolveMapping(nextMapping);
		},
		
		_prev : function(screen){
			var prevMapping = this.mappings[screen].previous;
			return this.resolveMapping(prevMapping);
		},
		
		resolveMapping : function(mappingRules){
			
			// null or litteral strings are straightforward
			if (mappingRules === null || typeof mappingRules === "string"){
				return mappingRules;
			}
			
			// in case of an object we must look further and apply
			// the condition on the model
			var res = null;
			var mega = this.mega;
			
			for (var rule in mappingRules){
				if (rule === "else") continue;
				var split = rule.split('=');
				var branchAttr = split[0],
					condition = split[1];
				
				if (mega.findNested(branchAttr) === condition){
					res = mappingRules[rule];
					break;
				}
				
			}
			
			return (res !== null) ? res : mappingRules["else"];
		},
						
		
		labels : function(){
			if (!this._labels){
				// lazy loading of the labels
				var labels = {};
				['create', 'next', 'previous', 'add'].forEach(function(lbl){ 
					labels[lbl] = 'henix.jirasync.execplan.designer.buttons.'+lbl;
				});				
				this._labels = translator.get(labels);
			}
			return this._labels;
		},
		
		
		getActiveButtons : function(){
			var BTNS = this.__SCREEN_BUTTONS();
			
			var btns;
			var labels = this.labels();
			
			
			switch(this.activeScreen){
			//the entry stage
			case this.firstScreen :
				btns = BTNS.START;
				break;
			
			// 'testcase-select' may be a final stage or an intermediate stage
			// depending on the route
			case "testcase-select" :
				var next = this._next('testcase-select');
				btns = (next === 'done') ? BTNS.ADD : BTNS.INTERMEDIATE;
				break;
			
			// 'iteration' is the other the final stage
			case "iteration-create": 
				btns = BTNS.CREATE;
				break;
			
			// the 'done' calls for a complete revalidation.
			// in case of error, just display the back button
			// if all is green, no button are returned (the rest 
			// of the operation is automatic)	
			case "done" : 
				this.refreshStatus();
				var doneStatus = this.screenMap['done'].status;

				btns = (doneStatus === 'COMPLETE') ? [] : BTNS.BACK;
				
				break;
			
			// for all other cases until specified otherwise
			default : 
				btns = BTNS.INTERMEDIATE;
			}
			
			return btns;
			
		},
		
		// support for 'getActiveButtons'
		__SCREEN_BUTTONS : function(){
			
			var labels = this.labels();
			
			return {
				'START' : [{
					label : labels.next,
					event : 'advance'
				}],
				
				'INTERMEDIATE' : [{
					label : labels.next,
					event : 'advance'
				},{
					label : labels.previous,
					event : 'back'
				}],
				
				'CREATE' : [{
					label : labels.create,
					event : 'go'
				},{
					label : labels.previous,
					event : 'back'
				}],
				
				'ADD' : [{
					label : labels.add,
					event : 'go'
				},{
					label : labels.previous,
					event : 'back'
				}],
				
				'BACK' : [{
					label : labels.previous,
					event : 'back'
				}]
			};
		}
		
	});
	
	return ScreenFlow;
	
});


		