/*
 *     This file is part of the Jira Agile synchronization connector for Squash TM (plugin.requirement.xsquash4jira) project.
 *     Copyright (C) 2017 - 2019 Henix, henix.fr - All Rights Reserved
 *
 *     Unauthorized copying of this file, via any medium is strictly prohibited
 *     Proprietary and confidential
 *
 * 	 (C)Henix. Tous droits réservés.
 *
 * 	Avertissement : ce programme est protégé par la loi relative au droit d'auteur et par les conventions internationales. Toute reproduction ou distribution partielle ou totale du logiciel, par quelque moyen que ce soit, est strictement interdite.
 */
/*
 * This is the base class for our backbone subviews. See JirasyncExecplanMainView for an introduction and the 
 * big picture.
 * 
 *  ---------------
 *  Name 
 *  ---------------
 *  
 *  Each subview must declare a name, because that's how they will be 
 *  identified by the rest of the system.
 *  
 *  ---------------
 *  Rendering
 *  ---------------
 *  
 *  The subviews must declare the template using its css selector.
 *  
 *  They can also declare the method templateModel, which are the arguments
 *  passed to Handlebars for the template interpolation.
 *  
 *  If there are complex widgets in the page, subviews must destroy them 
 *  properly by implementing _destroyWidgets().
 *  
 *  The subscreens can benefit from the predefined methods : 
 *  
 *  __renderWait() : renders the waiting pane
 *  __renderInvalid() : renders the invalid pane
 *  __renderValid() : renders the nominal template with the model given by templateModel() 
 *  
 *  
 *  ---------------
 *  Screen Status
 *  ---------------
 *  
 *  The status of a screen drives what it displays and what is shown in the 
 *  sidebar. 
 *  
 *  The screen status at any given time is quite self-explanatory in method #_computeStatus() 
 *  
 *  Implementors can override isReady() and isComplete(),
 *  that respectively tells whether the screen is either ready or complete.
 *  
 * 
 * ----------------
 * Model events
 * ----------------
 * 
 * A BaseScreen and subclasses are mixed with the ModelEventMixin. A hard-wired model
 * event is request for rendering, which is usually triggered when it is invalidated 
 * du to external factors. 
 * 
 * The event is 'model-invalidated' and the handler is 'onModelInvalidated'. The 
 * default behavior is to set the requireRendering to true and forward to the main
 * view, implementors may override this if they decide otherwise.
 *   
 * 
 */
define(["jquery", "handlebars", "underscore", "backbone", "./ModelEventsMixin", "squash.translator", "./error-handler", "./handlebars-helpers"], 
		function($, Handlebars, _, Backbone, ModelEventsMixin, translator, errHandler){
	

	/*
	 * Pseudo enum for the screen status. The semantic is :
	 * 
	 * - PREMATURE	: the step is reached but the the preconditions for the steps are not fulfilled.
	 * - READY		: the precondition for the steps are fullfiled, but the user has yet to complete the step 
	 * - COMPLETE 	: the step was visited once, was completed, and the content are still up to date.
	 * - NEEDS_REVIEW : the step was in a complete state but the user has changed something in the past in the mean time, 
	 * 					so it must recheck this screen.
	 */
	var ScreenStatus = {
		INITIAL		: "INITIAL",
		PREMATURE	: "PREMATURE",
		READY		: "READY",
		COMPLETE 	: "COMPLETE",
		NEEDS_REVIEW : "NEEDS_REVIEW",
		
		nominalStatus : ["READY", "COMPLETE", "NEEDS_REVIEW"],
		nonNominalStatus : ["INITIAL", "PREMATURE"]
	};
	
	
	var BaseScreen = Backbone.View.extend({
		
		name : "",

		template : undefined,
		
		
		// ***** internal state variables **********

		// that flag is true until the user visited the screen,
		// or when the model was invalidated
		needsReview : true,
		
		status : ScreenStatus.INITIAL,
		
		active : false,
		
		requireRendering : true,
		
		// ****************** init ****************
		
		initialize : function(options){
			var self = this;
			
			Backbone.View.prototype.initialize.apply(this, arguments);

			//listen to own model when it requires rendering
			//do not use modelEvents because subclasses may override it
			this.listenTo(options.model, 'model-invalidated', function(){
				self.onModelInvalidated();
			});
			

			// call to the ModelEventMixin
			this.bindModelEvents();
			
		},
		
		onModelInvalidated : function(){
			this.requireRendering = true;
			this.needsReview = true;
			this.$el.trigger('model-invalidated', {screen : this.name});
		},
		
		modelEvents : {
			// nothing by default
		},
		
		// ************* status management ***************
		
		setActive : function(active){
			this.active = active;
		},

		// When the status changed from non-nominal to nominal (and vice-versa), 
		// rendering is needed
		isRenderingNeeded : function(newStatus){
			
			var wasNominal = _.contains(ScreenStatus.nominalStatus, this.status);
			var isNominal = _.contains(ScreenStatus.nominalStatus, newStatus);
			
			return wasNominal !== isNominal;
			
		},

		refreshStatus : function(prevStatus){
			
			var newStatus = ScreenStatus.INITIAL;
			
			// whatever the new status : if the status of 
			// the previous screen is not complete,
			// this screen is premature
			if (prevStatus !== ScreenStatus.COMPLETE){
				newStatus = ScreenStatus.PREMATURE;
			}
			// else compute normally
			else{
				newStatus = this._computeStatus();
			}
						
			// now we have the new status,
			// let's decide if re-rendering is needed
			this.requireRendering = this.requireRendering || this.isRenderingNeeded(newStatus);
			
			// if the screen is both nominal and active,
			// consider that the content has been reviewed
			if (this.active && _.contains(ScreenStatus.nominalStatus, newStatus)){
				this.needsReview = false;
			}
			
			this.status = newStatus;
			
			// broadcast the new status
			this.$el.trigger('screenstatus-changed', {
				screen : this.name,
				status : newStatus
			});
			
			return this.status;
		},
		

		_computeStatus : function(){
			
			var isReady = this.isReady();
			var isComplete = this.isComplete();
			var needsReview = this.needsReview;
			
			var newStatus = ScreenStatus.INITIAL;
			
			if (isComplete && ! needsReview){
				newStatus = ScreenStatus.COMPLETE;
			}
			else if (isComplete && needsReview){
				newStatus = ScreenStatus.NEEDS_REVIEW;
			}
			else if (isReady){
				newStatus = ScreenStatus.READY;
			}
			else{
				newStatus = ScreenStatus.PREMATURE;
			}
	
			
			return newStatus;
			
		},

		/*
		 * Do the model have the required data to work ?
		 *  
		 * Implementors must override this if the model 
		 * can be not ready, for instance due to lack of 
		 * prerequisite information from the megamodel.
		 */
		isReady : function(){
			return true;
		},
		
		/*
		 * Has the user completed this step ?
		 * 
		 * Implementors must override this to check 
		 * if the model contains the information 
		 * the user is asked for. 
		 */
		isComplete : function(){
			return true;
		},
		
		// ************** external accessors *************
		
		show : function(){
			this.$el.show();
		},
		
		hide : function(){
			this.$el.hide();
		},
		
		addClass : function(classes){
			this.$el.addClass(classes);
		},
		
		removeClass : function(classes){
			this.$el.removeClass(classes);
		},
		
		// ***************** rendering etc ****************
				
		renderIfRequired : function(){
			if (this.requireRendering){
				this.renderIfOk();
			}
			this.requireRendering = false;
		},
		
		renderIfOk : function(){
			/*
			 * If the screen is not OK or accessed out of sequence,
			 * render the error message.
			 */
			if (this.status === "PREMATURE" || this.status === "INITIAL"){
				return this.__renderInvalid();
			}
			/*
			 * else (status is READY or COMPLETE), render it.
			 */
			else{
				return this.render();
			}
		},	
		
		render : function(){
			this.__renderValid();
		},
				
		remove : function(){
			this._destroyWidgets();
			BaseScreen.prototype.remove.apply(this, arguments);
		},
		
		_destroyWidgets : function(){
			// nothing for most screens
		},
		
		__renderWait : function(){
			this._destroyWidgets();
			this.$el.empty();
			var wait = Handlebars.compile($("#jirsync-plan-waitpane-template").html());
			this.$el.html(wait());
			return this;
		},
		
		__renderInvalid : function(){
			this._destroyWidgets();
			this.$el.empty();
			var nok = Handlebars.compile($("#jirsync-plan-precondfail-template").html());
			this.$el.html(nok());
			return this;
		},
		
		__renderValid : function(){
			this._destroyWidgets();
			this.$el.empty();
			var tpl = $(this.template).html();
			var tplFn = Handlebars.compile(tpl);
			var tplModel = this.templateModel();
			var html = tplFn(tplModel);
			this.$el.html(html);
			this.enableUntilBottom();
			return this;
		},
		
		__renderAjaxError : function(jqXhr){
			this._destroyWidgets();
			this.$el.empty();
			var nok = Handlebars.compile($("#jirsync-plan-ajaxerror-template").html());			
			var msg = errHandler.extractMessage(jqXhr.responseJSON);			
			this.$el.html(nok(msg));
			return this;			
		},
		
		// a variant that does remove the wait pane only when a given event is triggered. 
		__deferredRenderValid : function(eventName){
			var self = this;
			this.$el.one(eventName, function(){
				self.$el.find('.jirsync-waitpane').remove();
			});
			
			this._destroyWidgets();
			
			var tpl = $(this.template).html();
			var tplFn = Handlebars.compile(tpl);
			var tplModel = this.templateModel();
			var html = tplFn(tplModel);
			this.$el.append(html);
			this.enableUntilBottom();
			return this;
		},
		
		enableUntilBottom : function(){
			var elts = this.$el.find('.until-bottom');
			if (elts.length === 0) return;
			elts.each(function(){
				var elt = $(this);
				var eltTop = elt.position().top;
				
				// we'll need to take account for the margins, padding etc
				var eltMargins = elt.outerHeight(true) - elt.height();
				
				// now get the height of the parent (here, height means the content)
				var parent = elt.parent();
				
				// first check max-height if defined, if not available find height
				var parentHeight = parent.css('max-height');
				if (parentHeight !== 'none'){
					// extract how many pixels that makes
					parentHeight = /([\d\.]+)px/.exec(parentHeight)[1];
				}
				else{
					parentHeight = parent.height();
				}
							
				elt.css('max-height', (parentHeight-eltTop-eltMargins)+'px');
			});
			
		},
				
		templateModel : function(){
			return undefined;
		}
		
	});


	_.extend(BaseScreen.prototype, ModelEventsMixin);
	
	
	return BaseScreen;
	
	
});

