/*
 *     This file is part of the Jira Agile synchronization connector for Squash TM (plugin.requirement.xsquash4jira) project.
 *     Copyright (C) 2017 - 2019 Henix, henix.fr - All Rights Reserved
 *
 *     Unauthorized copying of this file, via any medium is strictly prohibited
 *     Proprietary and confidential
 *
 * 	 (C)Henix. Tous droits réservés.
 *
 * 	Avertissement : ce programme est protégé par la loi relative au droit d'auteur et par les conventions internationales. Toute reproduction ou distribution partielle ou totale du logiciel, par quelque moyen que ce soit, est strictement interdite.
 */
/*
 * Optional screens if the chosen strategy is BY_SPRINT.
 * 
 * Note that here we got two screens that share the same model (instead of one per model).
 * 
 */
define(["jquery",  "./BaseModel", "./BaseScreen", "./classes", "underscore", 
	"squash.configmanager", "squash.dateutils", "squash.translator", "app/util/StringUtil",
	"jqueryui", "jeditable", "jeditable.datepicker"], 
		function( $, BaseModel, BaseScreen, classes, _, confman, dateutils, translator, StringUtils){
	

	var dataService = squashtm.app.contextRoot + 'jirasync/exec-plan';
	

	
	// ******* strategy BY_SPRINT model **********
	
	
	var BySprintModel = BaseModel.extend({

		consumes : ['strategy'],
		produces : ['selectedSprints'],
		
		
		defaults : {
			// --- the ultimate goal of this model ----
			
			// this is the final result of the couple of screens 'board select' and 'sprint select'
			// (as indicated by the 'produces' array)
			selectedSprints : [],
			
			// --- local UI state data ----
			projectId : null,
			projectName : "",
			scope : "SQ_PROJ",		
			
			// manually selected JIRA Boards
			userJiraSelection : [],
			
			// filtering options
			optFilter : "NO_FILTER",
			optCompleteDate : null,				
			optMostRecentEnabled : false,
			optMostRecent : 1,
			optNameLikeEnabled : false,
			optNameLike : "",
			
			// ---- data fetched from the server -----
			
			// JIRA boards for the current squash project
			forSelectedProject : [],
			
			// all available JIRA projects
			forAllProjects : [],
			
			// the set of available sprints that match the selected boards and filters.
			// On screen 'Board selection', the number of sprints is displayed in the sprint counter.
			// On screen 'Sprint selection', the content is displayed in a list.
			sprints : []
			
		},
		
		init : function(mega){
			
			this.on({
				'change:scope' : this.refetchSprints,
				'change:userJiraSelection' : this.refetchSprints,
				'change:optFilter' : this.refetchSprints,
				'change:optCompleteDate' : this.refetchSprints,
				'change:optNameLikeEnabled' : this.refetchSprints,
				'change:optNameLike' : this.refetchSprints,
				'change:optMostRecentEnabled' : this.refetchSprints,
				'change:optMostRecent' : this.refetchSprints
			});
			
			return {
				projectId : mega.selectedNode().projectId,
				projectName : mega.selectedNode().projectName
			};
		},
		
		
		// ******************* flags *********************
		
		/*
		 * This flag helps ensuring that the boards are fetched only once
		 */
		flagFetchBoards : true,
		
		/*
		 * The sprint fetch postprocessing logic is driven by this flag.
		 * A value of "select-all" will call the 'postprocessSelectAll' behavior,
		 * while value "retain-selected" will call the 'postprocessRetainSelected' behavior
		 * 
		 * (see 'fetchSprintsIfNeeded')
		 */
		sprintPostprocessMode : "select-all",
		
		// this changes the postprocessing behavior
		setPostprocessRetainSelected : function(){
			this.sprintPostprocessMode = "retain-selected";
		},

		// ************ internal events handling *********
		
		refetchSprints : function(){
			/*
			 *  we need to 
			 *  1/ invalidate the attribute 'sprints' (this tells the 'select sprints' view to redraw)
			 *  2/ fetch the sprints immediately because we need to count them.
			 */ 
			this.invalidate();
			this.fetchSprintsIfNeeded();		
		},

		
		// ********** board selection part ***********

		// must return a promise
		fetchBoardsIfNeeded : function(){
			
			if (! this.flagFetchBoards){
				return $.Deferred().resolve.promise();
			}
			
			var self = this;
			var url = dataService + '/jira-boards?selected-squash-project-id='+this.attributes.projectId;
			
			return $.ajax({
				url : url,
				type : 'GET',
				dataType : 'json'
			})
			.done(function(json){
				self.forSelectedProject(json.forSelectedProject);
				self.forAllProjects(json.forAllProjects);
				self.flagFetchBoards = false;
				return this;
			});
		},
		
		// ************ sprints selection part **************
		
		getEffectiveBoardSelection : function(){
			var scope = this.scope();
			return (scope==="SQ_PROJ") ? this.forSelectedProject() : this.userJiraSelection();
		},
		
		// must also return a promise
		fetchSprintsIfNeeded : function(){
			
			if (this.isFresh()){
				return $.Deferred().resolve().promise();
			}
			
			var self = this;
			var url = dataService + '/jira-boards/sprints/search';
			
			var payload = this.prepareFetchSprintPayload();
			
			return $.ajax({
				url : url,
				type : 'post',
				data : JSON.stringify(payload),
				contentType : 'application/json',
				dataType : 'json'
			})
			.done(function(json){
				self.sprints(json.result);
				
				// postprocessing
				if (self.sprintPostprocessMode === "select-all"){
					self.postprocessSelectAll();
				}
				else{
					self.postprocessRetainSelected();
				}
				
				self.fresh = true;				
				
				// force the triggering of 'change:sprints' otherwise the BoardsSelection 
				// screen will have a problem
				self.trigger('change:sprints');
				
				return this;
			});
		},
		
		prepareFetchSprintPayload : function(){
			var boards = this.getEffectiveBoardSelection();
			var complSince = (this.optFilter() === "COMPLETED_ON") ? this.optCompleteDate() : null;
			var stillRun = (this.optFilter() === "STILL_RUNNING");
			
			var nameLike = (this.optNameLikeEnabled()) ? this.optNameLike() : null;
			var mostRecent = (this.optMostRecentEnabled()) ? this.optMostRecent() : null;
			
			return {
				boards : boards,
				completedSince : complSince,
				nameLike : nameLike,
				mostRecentReleasesLimit : mostRecent,
				sprintStillRunning : stillRun
			};
		},
		
		/*
		 * This postprocessing behavior will simply consider 
		 * that all newly fetched sprints are selected 
		 */
		postprocessSelectAll : function(){
			this.selectedSprints(this.sprints());
		},
		
		/*
		 * This postprocessing behavior will compare the existing 
		 * selection with the new set of available sprints
		 * and retain selected only those that are still present.
		 */
		postprocessRetainSelected : function(){
			// turning the BoardSprint to a string form 
			// will greatly simplify the impact calculus
			var stockAsString = classes.BoardSprints.toStrings(this.sprints());
			var selecAsString = classes.BoardSprints.toStrings(this.selectedSprints());
			
			// using the string form, remove any selected item that is not available anymore
			// in the new stock of item
			var retainedAsString = _.filter(selecAsString, function(selected){
				return _.contains(stockAsString, selected);
			});
			
			// has anything been removed in the process ?
			// if so, set the new value in the model 
			if (retainedAsString.length < selecAsString.length){
				var parsed = classes.BoardSprints.fromStrings(retainedAsString);
				this.selectedSprints(parsed);
			}		
		}
		
	});
	
	
	
	// **** By Sprints Strategy : Board selection **********

	var BoardSelectionScreen = BaseScreen.extend({
		
		name : "board-select",
		
		el : "#screen-board-select",
		
		template : "#jirsync-plan-bysprint-boardsel-template",
		
		userDformat : null,		
		storageDformat : dateutils.ISO_8601,
		
		initialize : function(){
			BaseScreen.prototype.initialize.apply(this, arguments);
			this.userDformat = translator.get('squashtm.dateformatShort.datepicker');
		},
				
		
		// ********** status *********** 
		
		isComplete : function(){
			return this.model.getEffectiveBoardSelection().length > 0;
		},
		
		onModelInvalidated : function(){
			// the model is currently refetching the available sprints : 
			// put the counter in wait mode
			this.setSprintCounterWaitMode();
		},
		
		
		// ******** rendering **********
		
		_destroyWidgets : function(){
			this.completeDatePicker().editable('destroy');
		},
		
		render : function(){
			var self = this;

			this.__renderWait();
			this.model.fetchBoardsIfNeeded()
				.done(function(){
					// TODO : rewrite this with proper promise chaining
					BaseScreen.prototype.__renderValid.apply(self, arguments);
					self.initCompleteDatePicker();
					// fetching the sprints now will update the counter
					self.model.fetchSprintsIfNeeded()
						.done(function(){
							self.setSprintCounterDisplayMode()
						});
				});
			
			return this;
			
		},
		
		initCompleteDatePicker : function(){		
			
			var datepicker = this.completeDatePicker();
			var pickerconf = confman.getStdDatepicker();
			var jedconf = confman.getStdJeditable();
			
			var self = this;
			
			datepicker.editable(
				function(value){					
					var date = datepicker.find('input').datepicker('getDate');
					var asString = dateutils.format(date, this.storageDformat);
					datepicker.trigger('change', {value : asString});
					return value;
				}, {
				type : "datepicker",
				datepicker : pickerconf
			});
			
			datepicker.editable("disable");
			
		},
		
		
		templateModel : function(){
			
			var scope = this.model.scope();
			var selected = this.model.userJiraSelection();
			
			var mostRecentEnabled = this.model.optMostRecentEnabled();
			var mostRecent = this.model.optMostRecent();

			var nameLikeEnabled = this.model.optNameLikeEnabled();
			var nameLike = this.model.optNameLike();
			
			var subpaneEnabled = (scope === "ALL_BOARD");
			
			return {
				radiocheck : scope,
				
				sqProjName : this.model.projectName(),
				forThisSquashProject : this.model.forSelectedProject().join(','),
				
				subpane : {
					enabled : subpaneEnabled,
					boards : this.model.forAllProjects().map(function(proj){
						return {
							name : proj,
							checked : _.contains(selected, proj),
							disabled : !subpaneEnabled
						}
					})
				},
			
				opts : {
					mostRecent : {
						checked : mostRecentEnabled,
						limit : mostRecent,
						disabled : ! mostRecentEnabled
					},
					nameLike : {
						checked : nameLikeEnabled,
						name : nameLike,
						disabled : ! nameLikeEnabled
					}
				}
				
			};
		},


		
		// *********** state mutators *******************

		events : {
			'change input[name="bysprint-scope"]' : 'changeScope',
			'change input[name="bysprint-filter"]' : 'changeFilter',
			'change .jirsync-complete-date-picker' : "changeCompleteDate",
			'change .jirsync-subpane input' : 'changeSelection',

			'change .jirsync-mostrecent-check' : 'enableMostRecent',
			'change .jirsync-mostrecent-input' : 'changeMostRecent',

			'change .jirsync-namelike-check' : 'enableNameLike',
			'change .jirsync-namelike-input' : 'changeNameLike',
		},
		
		modelEvents : {
			'change:sprints' : "setSprintCounterDisplayMode"
		},

		changeScope : function(evt){
			var value = $(evt.currentTarget).val();
			this.model.scope(value);
			
			this.enableSubpane();
		},
		
		changeSelection : function(){
			var projects = this.subpane()
								.find('input:checked')
								.map(function(){
									return this.value;
								})
								.get();
			this.model.userJiraSelection(projects);
		},
		
		changeFilter : function(evt){
			var val = $(evt.currentTarget).val();
			this.model.optFilter(val);
			
			// also, enable/disable the datepicker, and the pseudo style
			var dp = this.completeDatePicker(); 
			if (val === "COMPLETED_ON"){
				dp.editable("enable");
				dp.removeClass('jirsync-disabled');
			}
			else{
				dp.editable("disable");
				dp.addClass('jirsync-disabled');
				
			}
		},
		
		changeCompleteDate : function(evt, value){
			this.model.optCompleteDate(value.value);
		},
		
		enableMostRecent : function(evt, value){
			var enabled = $(evt.currentTarget).prop('checked');
			this.model.optMostRecentEnabled(enabled);
			
			this.$el.find('.jirsync-mostrecent-input').prop('disabled', !enabled);
		},
		
		changeMostRecent : function(evt, value){
			var input = $(evt.currentTarget);
			var value = input.val();
			if (StringUtils.isBlank(value) || isNaN(value)){
				input.next().show();
			}
			else{
				this.model.optMostRecent(parseInt(value,10));
				input.next().hide();
			}
		},
		
		enableNameLike : function(evt, value){
			var enabled = $(evt.currentTarget).prop('checked');
			this.model.optNameLikeEnabled(enabled);
			
			this.$el.find('.jirsync-namelike-input').prop('disabled', !enabled);
		},
		
		changeNameLike : function(evt, value){
			var value = $(evt.currentTarget).val();
			this.model.optNameLike(value);
		},
		
		// ******** view elements **************************
		
		completeDatePicker : function(){
			return this.$el.find('.jirsync-complete-date-picker');
		},
		
		subpane : function(){
			return this.$el.find('.jirsync-subpane');
		},
		
		enableSubpane : function(){
			var scope = this.model.scope();
			var enabled = (scope === "ALL_BOARD");
			
			var opacity = enabled ? 1.0 : 0.3;
			var chkDisabled = !enabled;
			
			var subpane = this.$el.find('.jirsync-subpane');
			subpane.css('opacity', opacity);
			subpane.find('input').prop('disabled', chkDisabled);
		},
		
		setSprintCounterWaitMode : function(){
			this.$el.find('.source-counter-wait').show();
			this.$el.find('.source-counter-message').hide();
		},
		
		setSprintCounterDisplayMode : function(){
			this.$el.find('.source-counter-wait').hide();
			this.$el.find('.source-counter-message').show();
			
			var cnt = _.chain(this.model.sprints())
						.map(function(board){ return board.sprints})
						.flatten()
						.value()
						.length;
			this.$el.find('.source-counter').text(cnt);
		}
		
		
	});
	
	
	
	// **** By Sprint Strategy : Sprint selection **********
	
	var SprintSelectionScreen = BaseScreen.extend({
		
		name : "sprint-select",
		
		el : "#screen-sprint-select",
		
		template : "#jirsync-plan-bysprint-sprintsel-template",
		
		render : function(){
			var self = this;

			this.__renderWait();
			this.model.fetchSprintsIfNeeded()
				.done(function(){
					BaseScreen.prototype.__renderValid.apply(self, arguments);
				});
			return this;
		},
		
		templateModel : function(){			
			
			var selecAsString = classes.BoardSprints.toStrings(this.model.selectedSprints());
			
			var sprintModel = this.model.sprints().map(function(boardspr){
				var board = boardspr.board;
				var sprints = boardspr.sprints.map(function(sprint){
					var asString = board+"-"+sprint.id;
					var checkmix = {
						checked : _.contains(selecAsString, asString)
					};
					return _.extendOwn(checkmix, sprint);
					
				});
				return new classes.BoardSprints(board, sprints);
			});
			
			return {
				sprints : sprintModel
			};						
		},
		
		// ************* status *******************
		
		isReady : function(){
			var selection = this.model.getEffectiveBoardSelection();
			
			return selection.length > 0;
		},
		
		/*
		 * This screen is complete if :
		 *  - the user has selected something and 
		 * 	- the model was not invalidated in the mean time
		 */
		isComplete : function(){
			var selection = this.model.selectedSprints();
			
			return selection.length > 0 ;
		},
		
		// ************* events *******************
		
		events : {
			'change input' : 'changeSprints',
			'click .sprintsel-select' : 'groupSelect'
		},
		
		changeSprints : function(){
			
			var checkboxes = this.$el.find('input:checked');
			
			// parse and map the selection as a map<string,list<long>> in javaspeak
			var asHash = {};
			checkboxes.each(function(){
				var sprintId = this.value;
				var board = $(this).parents('ul:first').data('board');
				
				asHash[board] = asHash[board] || [];
				asHash[board].push({ id : sprintId});
			});
			
			// now convert as boardSprints
			var boardsprints = Object.keys(asHash).map(function(entry){
				var board = entry;
				var sprints = asHash[entry];
				return new classes.BoardSprints(board, sprints);
			});
			
			this.model.selectedSprints(boardsprints);
			
			// also, now the user has interacted with the model,
			// we need to instruct it to change its sprint fetch postprocessing behavior
			this.model.setPostprocessRetainSelected();
		},
		
		groupSelect : function(evt){
			// define what to do with the checkboxes
			var mode = $(evt.currentTarget).data('select');
			var operation;
			switch(mode){
			case 'all' : operation = function(){this.checked=true}; break;
			case 'none' : operation = function(){this.checked=false;}; break;
			case 'invert' : operation = function(){this.checked = !this.checked;}; break;
			default : throw "unsupported selection operation : "+mode;
			}
			
			// apply the operation
			this.$el.find('input[type="checkbox"]').each(operation);
			
			// now save the new selection
			this.changeSprints();
		}
		
	});
	
	
	return {
		model : BySprintModel,
		brdselScreen : BoardSelectionScreen,
		sprselScreen : SprintSelectionScreen
	};
	
});