/*
 *     This file is part of the Jira Agile synchronization connector for Squash TM (plugin.requirement.xsquash4jira) project.
 *     Copyright (C) 2017 - 2019 Henix, henix.fr - All Rights Reserved
 *
 *     Unauthorized copying of this file, via any medium is strictly prohibited
 *     Proprietary and confidential
 *
 * 	 (C)Henix. Tous droits réservés.
 *
 * 	Avertissement : ce programme est protégé par la loi relative au droit d'auteur et par les conventions internationales. Toute reproduction ou distribution partielle ou totale du logiciel, par quelque moyen que ce soit, est strictement interdite.
 */
define(["./BaseModel", "underscore", "app/util/StringUtil", "moment"], function(BaseModel, _, StringUtils, moment){
	
	
	/*
	 * Attributes match those of the java class IterationFormModel,
	 * 
	 * with the additional attribute 'meta' that contains everything 
	 * needed for validation.
	 */
	var IterationFormModel = BaseModel.extend({
		
		defaults : {
			name : '',
			reference : '',
			description : '',
			
			// that one is a hash of { <id> : <value>}
			customFields : {},
			
			
			// this attribute comes from java class IterationFormMetadata 
			meta : {
				
				siblingNames : [],
				
				// that one is the list of all custom field definitions
				customFields : [] 
			}
				
		},
		
		
		toJSON : function(){
			return _.omit(this.attributes, 'meta');				
		},
		
		//*************** validation ***************
		errorCodes : {
			NOT_BLANK : "NOT_BLANK",
			DUPLICATE : "DUPLICATE",
			INVALID_FORMAT : "INVALID_FORMAT",
			NOT_A_NUMBER : "NOT_A_NUMBER"
		},
		
		validate : function(attrs){
			var self = this,
				attrs = attrs || this.attributes;
			var errors = {};
			
			// 1 - name not blank nor equal to any siblings 
			if (StringUtils.isBlank(attrs.name)){
				errors['name'] = this.errorCodes.NOT_BLANK; 
			}
			
			if (_.contains(attrs.meta.siblingNames, attrs.name)){
				errors['name'] = this.errorCodes.DUPLICATE;
			}
			
			
			// custom fields now
			var cufDefs = attrs.meta.customFields;
			
			_.where(attrs.meta.customFields, {optional : false})
				.forEach(function(cuf){

					// 2 - non optional custom fields must not be empty
					var id = cuf.id;
					var value = attrs.customFields[id];
					
					if ((value === null)	|| 
						(value === undefined)	||
						(_.isString(value) && StringUtils.isBlank(value))	||
						(_.isArray(value) && value.length === 0)){
						errors['customFields-'+id] = self.errorCodes.NOT_BLANK;
					}
					
					// 3 - specific data formats that can fail
					// 3.1 - numeric field must be a number
					var cuf = _.findWhere(cufDefs, { id : id });
					
					switch(cuf.inputType.enumName){
					case 'NUMERIC' : 
						if ( isNaN(value) ){
							errors['customFields-'+id] = self.errorCodes.NOT_A_NUMBER;
						}
						break;
						
					case 'DATE_PICKER' : 
						if (! moment(value, "YYYY-MM-DD", true).isValid()){
							errors['customFields-'+id] = self.errorCodes.INVALID_FORMAT;
						}
						
					default : break;
					}
					
					
					
				});
			
			
			// That's all for now, I trust the customfield accessors 
			// to validate themselves their values.
			// If needed add more code here some day.
			
			return (!_.isEmpty(errors)) ? errors : undefined;
			
		}
		
	});
	
	return IterationFormModel;
	
});